///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//  Copyright  NetworkDLS 2002, All rights reserved
//
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF 
// ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO 
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A 
// PARTICULAR PURPOSE.
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef _NSWFL_WINSFL_CPP_
#define _NSWFL_WINSFL_CPP_
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include <Windows.H>
#include <StdIO.H>
#include <StdLib.H>
#include <ShlObj.H>

#include "NSWFL.H"

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void CopyTextBoxToClipboard(HWND hWnd)
{
	SendMessage(hWnd, EM_SETSEL, (WPARAM)0, (LPARAM)-1);
	SendMessage(hWnd, WM_COPY, (WPARAM)0, (LPARAM)0);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void TextBoxInsert(HWND hWnd, const char *sInBuf)
{
    LRESULT iLen = SendMessage(hWnd, WM_GETTEXTLENGTH, 0, 0);
    SendMessage(hWnd, EM_REPLACESEL, 0, (LPARAM)(LPSTR)sInBuf);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void TextBoxWrite(HWND hWnd, const char *sInBuf)
{
    LRESULT iLen = SendMessage(hWnd, WM_GETTEXTLENGTH, 0, 0);
    SendMessage(hWnd, EM_SETSEL, (WPARAM)iLen, (LPARAM)iLen);
    SendMessage(hWnd, EM_REPLACESEL, 0, (LPARAM)(LPSTR)sInBuf);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void MsgBox(const __int64 i64Value)
{
	char sValue[64];
	sprintf_s(sValue, sizeof(sValue), "%i", i64Value);
    MessageBox(GetActiveWindow(), sValue, "Message", MB_TASKMODAL);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void MsgBox(const int iValue)
{
	char sValue[64];
	sprintf_s(sValue, sizeof(sValue), "%d", iValue);
    MessageBox(GetActiveWindow(), sValue, "Message", MB_TASKMODAL);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void MsgBox(const float fValue)
{
	char sValue[1024];
	sprintf_s(sValue, sizeof(sValue), "%f", fValue);
    MessageBox(GetActiveWindow(), sValue, "Message", MB_TASKMODAL);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void MsgBox(const double dValue)
{
	char sValue[1024];
	sprintf_s(sValue, sizeof(sValue), "%f", dValue);
    MessageBox(GetActiveWindow(), sValue, "Message", MB_TASKMODAL);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void MsgBox(const char *sValue)
{
    MessageBox(GetActiveWindow(), sValue, "Message", MB_TASKMODAL);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void ErrorMessage(HWND hWnd, const char *sInBuf)
{
    HWND hOwner = NULL;

    if(hWnd == NULL)
	{
		hOwner = GetActiveWindow();
	}
    else hOwner = hWnd;

    MessageBox(hOwner, sInBuf ,"Error", MB_ICONHAND + MB_SYSTEMMODAL + MB_TASKMODAL);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void AlertMessage(HWND hWnd, const char *sInBuf, const char *sTitle)
{
    HWND hOwner = NULL;

    if(hWnd == NULL)
	{
		hOwner = GetActiveWindow();
	}
    else hOwner = hWnd;

    MessageBox(hOwner, sInBuf, sTitle, MB_SYSTEMMODAL + MB_TASKMODAL);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int Get_TextLength(HWND hWnd)
{
    return GetWindowTextLength(hWnd);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

double Get_Double(HWND hWnd)
{
    char sBuf[16];
    Get_Text(hWnd, sBuf, sizeof(sBuf));
	return dFormattedValue(sBuf);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

float Get_Float(HWND hWnd)
{
	return (float)Get_Double(hWnd);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

__int64 Get_Int64(HWND hWnd)
{
    char sBuf[16];
    Get_Text(hWnd, sBuf, sizeof(sBuf));
	return i64FormattedValue(sBuf);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

long Get_Long(HWND hWnd)
{
    char sBuf[16];
    Get_Text(hWnd, sBuf, sizeof(sBuf));
	return (long)i64FormattedValue(sBuf);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int Get_Int(HWND hWnd)
{
    return (int) Get_Long(hWnd);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void Set_LongFormatted(HWND hWnd, long lInput)
{
	char sBuf[64];
	FormatInteger(sBuf, sizeof(sBuf), lInput);
    Set_Text(hWnd, sBuf);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void Set_Long(HWND hWnd, long lInput)
{
	char sBuf[64];
	sprintf_s(sBuf, sizeof(sBuf), "%d", lInput);
    Set_Text(hWnd, sBuf);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void Set_Double(HWND hWnd, double dInput, int iDecimals)
{
	char sFmt[64];
	sprintf_s(sFmt, sizeof(sFmt), "%%.%df", iDecimals);

	char sBuf[256];
	sprintf_s(sBuf, sizeof(sBuf), sFmt, dInput);
    Set_Text(hWnd, sBuf);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void Set_Float(HWND hWnd, float fInput, int iDecimals)
{
	char sFmt[64];
	sprintf_s(sFmt, sizeof(sFmt), "%%.%df", iDecimals);

	char sBuf[256];
	sprintf_s(sBuf, sizeof(sBuf), sFmt, fInput);
    Set_Text(hWnd, sBuf);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void Set_DoubleFormatted(HWND hWnd, double dInput, int iDecimals)
{
	char sNumber[256];
	FormatDouble(sNumber, sizeof(sNumber), dInput, iDecimals);
    Set_Text(hWnd, sNumber);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void Set_FloatFormatted(HWND hWnd, float fInput, int iDecimals)
{
	char sNumber[256];
	FormatFloat(sNumber, sizeof(sNumber), fInput, iDecimals);
    Set_Text(hWnd, sNumber);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void Set_Int(HWND hWnd, int iInput)
{
    Set_Long(hWnd, (DWORD)iInput);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void Set_Text(HWND hWnd, const char *sInBuf)
{
    SendMessage(hWnd, (UINT)WM_SETTEXT, (WPARAM)0, (LPARAM)sInBuf);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//The pointer returned through [sOutBuf] must be freed externally.
int Get_Text(HWND hWnd, char *&sOutBuf)
{
	int iLength = Get_TextLength(hWnd);
	if(iLength > 0)
	{
		sOutBuf = (char *) calloc(sizeof(char), iLength + 1);
	    return GetWindowText(hWnd, sOutBuf, iLength + 1);
	}
	return 0;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int Get_Text(HWND hWnd, char *sOutBuf, int iMaxSize)
{
    int iLen = GetWindowTextLength(hWnd);

	if(iLen >= iMaxSize)
	{
		return iLen;
	}

    iLen = GetWindowText(hWnd, sOutBuf, iMaxSize);

    sOutBuf[iLen] = '\0';

    return iLen;
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void CenterOverOwner(HWND hWnd)
{
	CenterOverWindow(hWnd, GetParent(hWnd));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void CenterOverWindow(HWND hWnd, HWND hOtherWindow)
{
	if(hOtherWindow == NULL)
	{
		CenterWindow(hWnd);
		return;
	}

	int iTaksbarOffsetX = 0;
	int iTaksbarOffsetY = 0;

    int iScreenX = GetSystemMetrics(SM_CXSCREEN) - 10;
    int iScreenY = (GetSystemMetrics(SM_CYSCREEN) - GetSystemMetrics(SM_CYCAPTION)) - 10;

	MONITORINFO MonitorInfo;
	memset(&MonitorInfo, 0, sizeof(MonitorInfo));

	HMONITOR hMonitor = MonitorFromWindow(hWnd, MONITOR_DEFAULTTONEAREST);
	if(hMonitor)
	{
		MonitorInfo.cbSize = sizeof(MonitorInfo);

		if(GetMonitorInfo(hMonitor, &MonitorInfo))
		{
			iScreenX = MonitorInfo.rcMonitor.right - MonitorInfo.rcMonitor.left;
			iScreenY = MonitorInfo.rcMonitor.bottom - MonitorInfo.rcMonitor.top;

			//Figure the height/width of the taskbar.
			int iTaskScreenX = GetSystemMetrics(SM_CXSCREEN);
			int iTaskScreenY = (GetSystemMetrics(SM_CYSCREEN) - GetSystemMetrics(SM_CYCAPTION));
			int iTaskFullScreenX = GetSystemMetrics(SM_CXFULLSCREEN);
			int iTaskFullScreenY = (GetSystemMetrics(SM_CYFULLSCREEN) - GetSystemMetrics(SM_CYCAPTION));

			iTaksbarOffsetX = abs(iTaskScreenX - iTaskFullScreenX);
			iTaksbarOffsetY = abs(iTaskScreenY - iTaskFullScreenY);
		}
		else{
			memset(&MonitorInfo, 0, sizeof(MonitorInfo));
		}
	}

	RECT recOther;
	memset(&recOther, 0, sizeof(recOther));
	GetWindowRect(hOtherWindow, &recOther);

	RECT recDlg;
	memset(&recDlg, 0, sizeof(recDlg));
	GetWindowRect(hWnd, &recDlg);

	int iX = ((recOther.right - ((recOther.right-recOther.left) / 2)) - ((recDlg.right-recDlg.left)/2));
	int iY = ((recOther.bottom - ((recOther.bottom-recOther.top) / 2)) - ((recDlg.bottom-recDlg.top)/2));

	if(iX - MonitorInfo.rcMonitor.left > (iScreenX - (recDlg.right-recDlg.left)) - iTaksbarOffsetX)
	{
		iX = ((MonitorInfo.rcMonitor.left + (iScreenX - (recDlg.right-recDlg.left))) - iTaksbarOffsetX) - 10;
	}
	else if(iX - MonitorInfo.rcMonitor.left < 10)
	{
		iX = (MonitorInfo.rcMonitor.left + 10);
	}

	if(iY - MonitorInfo.rcMonitor.top > (iScreenY - (recDlg.bottom-recDlg.top)) - iTaksbarOffsetY)
	{
		iY = (MonitorInfo.rcMonitor.top + (iScreenY - (recDlg.bottom-recDlg.top)) - iTaksbarOffsetY) - 10;
	}
	else if(iY - MonitorInfo.rcMonitor.top < 10)
	{
		iY = MonitorInfo.rcMonitor.top + 10;
	}

	SetWindowPos(hWnd, NULL, iX, iY, 0, 0, SWP_NOSIZE|SWP_NOZORDER);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void CenterWindow(HWND hWnd)
{
	POINT MousePos;
	memset(&MousePos, 0, sizeof(MousePos));

	int iScreenX = GetSystemMetrics(SM_CXSCREEN);
	int iScreenY = GetSystemMetrics(SM_CYSCREEN);

	int MonitorOffsetX = 0;
	int MonitorOffsetY = 0;

	if(GetCursorPos(&MousePos))
	{
		MONITORINFO MonitorInfo;
		HMONITOR hMonitor = MonitorFromPoint(MousePos, MONITOR_DEFAULTTONEAREST);
		if(hMonitor)
		{
			memset(&MonitorInfo, 0, sizeof(MonitorInfo));
			MonitorInfo.cbSize = sizeof(MonitorInfo);

			if(GetMonitorInfo(hMonitor, &MonitorInfo))
			{
				MonitorOffsetX = MonitorInfo.rcMonitor.left;
				MonitorOffsetY = MonitorInfo.rcMonitor.top;

				iScreenX = MonitorInfo.rcMonitor.right - MonitorInfo.rcMonitor.left;
				iScreenY = MonitorInfo.rcMonitor.bottom - MonitorInfo.rcMonitor.top;
			}
		}
	}

    RECT wRect;
	memset(&wRect, 0, sizeof(wRect));

    GetWindowRect(hWnd, &wRect);

    DWORD x = ((iScreenX -(wRect.right-wRect.left))/2) + MonitorOffsetX;
    DWORD y = ((iScreenY-(wRect.bottom-wRect.top+GetSystemMetrics(SM_CYCAPTION)))/2) + MonitorOffsetY;

    SetWindowPos(hWnd,NULL,x,y,0,0,SWP_NOSIZE|SWP_NOZORDER);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

LRESULT CALLBACK Set_Color(int iFGColor, int iBKColor, WPARAM wParam)
{
    static HBRUSH ReUsableBrush;
    DeleteObject(ReUsableBrush);
    ReUsableBrush = CreateSolidBrush(iBKColor);
    SetTextColor((HDC)wParam,iFGColor);
    SetBkColor((HDC)wParam,iBKColor);
    return (LRESULT)ReUsableBrush;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function returns HWND of the child window under the mouse cursor.
		The window must be visible and uncovered.
*/
HWND GetWindowUnderPoint(void)
{
    POINT Pointer;
	memset(&Pointer, 0, sizeof(Pointer));
    if(GetCursorPos(&Pointer))
	{
		return WindowFromPoint(Pointer);
	}
	return NULL;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function returns TRUE if the mouse cursor is over the window specified by HWND.
		The window must be visible and uncovered.
*/
bool IsMouseOverWindow(HWND hWnd)
{
    POINT Pointer;
	memset(&Pointer, 0, sizeof(Pointer));
    GetCursorPos(&Pointer);
	return(WindowFromPoint(Pointer) == hWnd);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function returns TRUE if the mouse cursor is over the window specified by HWND.
		This function does not care if the window is covered, enabled or visible.
*/
bool IsMouseOverHwnd(HWND hWnd)
{
    RECT Rectangle;
    POINT Pointer;
    memset(&Rectangle, 0, sizeof(Rectangle));
    memset(&Pointer, 0, sizeof(Pointer));

    GetWindowRect(hWnd, &Rectangle);
    GetCursorPos(&Pointer);

    return(PtInRect(&Rectangle, Pointer) > 0);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool IsMouseOverhMenu(HWND hWnd, HMENU hMenu, int iMenuItem)
{
    RECT Rectangle;
    POINT Pointer;
    memset(&Rectangle, 0, sizeof(Rectangle));
    memset(&Pointer, 0, sizeof(Pointer));

	if(GetMenuItemRect(hWnd, hMenu, iMenuItem, &Rectangle))
	{
	    GetCursorPos(&Pointer);
	    return(PtInRect(&Rectangle, Pointer) > 0);
	}

	return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int OSFileDialog(HWND hOwner, char *sInOutFileName, int iMaxOutBufSz, char *sInDfltExt,
							char *sInTitle, char *sInFilters, char *sInDfltDir, int iDlgType, DWORD dwFlags)
{
    char sFileName[MAX_PATH];
    int iRetVal = 0;
	char *sCurrDir = NULL;

    OPENFILENAME OFN;

    if( (iDlgType != OFD) && (iDlgType != SFD) )
	{
		return 0;
	}

    strcpy_s(sFileName, sizeof(sFileName), sInOutFileName);

    OFN.lpstrCustomFilter = NULL;
    OFN.nMaxCustFilter    = 0;
    OFN.lpstrFileTitle    = NULL;
    OFN.nMaxFileTitle     = 0;

    if(!sInFilters)
	{
		OFN.lpstrFilter  = "All-Files\0*.*\0Text-Files\0*.txt\0\0";
	}
    else OFN.lpstrFilter  = sInFilters;

    if(!sInDfltExt)
	{
		OFN.lpstrDefExt = "txt";
	}
    else OFN.lpstrDefExt = sInDfltExt;

    if(!sInTitle)
    {
        if(iDlgType == OFD)
		{
			OFN.lpstrTitle = "Select file to open";
		}
        else if(iDlgType == SFD)
		{
			OFN.lpstrTitle = "Save file as";
		}
    }
    else OFN.lpstrTitle = sInTitle;

    if(!sInDfltDir)
	{
		if((sCurrDir = (char *) calloc(sizeof(char), MAX_PATH + 1)))
		{
			if(Get_CurrentDirectory(sCurrDir, MAX_PATH))
			{
				OFN.lpstrInitialDir = sCurrDir;
			}
		}
	}
    else OFN.lpstrInitialDir = sInDfltDir;

    OFN.lStructSize       = sizeof(OFN);
    OFN.hwndOwner         = hOwner;
    OFN.nFilterIndex      = 1;
    OFN.lpstrFile         = sFileName;
    OFN.nMaxFile          = sizeof(sFileName);

    if(dwFlags == 0)
	{
		OFN.Flags = OFN_EXPLORER + OFN_LONGNAMES + OFN_PATHMUSTEXIST;
	}
	else OFN.Flags = dwFlags;

	if(iDlgType == OFD)
	{
		iRetVal = GetOpenFileName(&OFN);
	}
	else if(iDlgType == SFD)
	{
		iRetVal = GetSaveFileName(&OFN);
	}

    strcpy_s(sInOutFileName, iMaxOutBufSz, sFileName);

	if(!sInDfltDir && sCurrDir)
	{
		free(sCurrDir);
	}

    return iRetVal;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int CALLBACK BrowseDirectoryCallbackProc(HWND hwnd, UINT uMsg, LPARAM lParam, LPARAM lpData)
{
	if(uMsg == BFFM_INITIALIZED)
	{
		SendMessage(hwnd, BFFM_SETSELECTION, (WPARAM)TRUE, lpData);
	}
	return 0;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//Assumes that sOutBuf is at least MAX_PATH, because SHGetPathFromIDList does not check for buffer size.
bool BrowseDirectory(HWND hOwner, char *sInTitle,
					 unsigned long ulFlags, char *sOutBuf, int iMaxOutBufSz)
{
	return BrowseDirectory(hOwner, sInTitle, ulFlags, sOutBuf, iMaxOutBufSz, NULL);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//Assumes that sOutBuf is at least MAX_PATH, because SHGetPathFromIDList does not check for buffer size.
bool BrowseDirectory(HWND hOwner, char *sInTitle,
					 unsigned long ulFlags, char *sOutBuf, int iMaxOutBufSz, char *sInitialDirectory)
{
    LPMALLOC pMalloc;
    LPITEMIDLIST lpItemIDList;
    BROWSEINFO  browseInfo;
    bool bResult = false;

	if(iMaxOutBufSz < MAX_PATH)
	{
		return false;
	}

    memset(&pMalloc,0,sizeof(pMalloc));
    memset(&browseInfo,0,sizeof(browseInfo));
    memset(&lpItemIDList,0,sizeof(lpItemIDList));

    if(S_OK != SHGetMalloc(&pMalloc))
	{
		return false;
	}

    memset(&browseInfo, 0, sizeof(BROWSEINFO));
    browseInfo.hwndOwner      = hOwner;
    browseInfo.pszDisplayName = sOutBuf;
    browseInfo.lpszTitle      = sInTitle;
    browseInfo.ulFlags        = ulFlags;

	if(sInitialDirectory)
	{
		browseInfo.lParam         = (LPARAM)sInitialDirectory;
		browseInfo.lpfn           = BrowseDirectoryCallbackProc;
	}

    lpItemIDList = SHBrowseForFolder(&browseInfo);

    if(lpItemIDList != NULL)
    {
        strcpy_s(sOutBuf, iMaxOutBufSz, "");

        if(SHGetPathFromIDList(lpItemIDList, sOutBuf))
        {
            if( strlen(sOutBuf) > 0 )
			{
				bResult = true;
			}
        }

        pMalloc->Free(lpItemIDList);
    }
    pMalloc->Release();

    return bResult;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

HFONT MakeFont(char *sFontName, int iPointSize)
{
    HDC hDC;
    int CyPixels = 0;
    hDC = GetDC(HWND_DESKTOP);
    CyPixels = GetDeviceCaps(hDC, LOGPIXELSY);
    ReleaseDC(HWND_DESKTOP, hDC);
    int iiPointSize = (iPointSize * CyPixels)/72;
    return CreateFont(0-iiPointSize, 0, 0, 0, FW_NORMAL, 0, 0, 0,
		ANSI_CHARSET, OUT_TT_PRECIS, CLIP_DEFAULT_PRECIS,
		DEFAULT_QUALITY, FF_DONTCARE, sFontName);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void SetFont(HWND hWnd, HFONT hFont)
{
    SendMessage(hWnd , WM_SETFONT, (WPARAM)hFont, (LPARAM)TRUE);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//This function will wait for the window specified by hWnd to become visible,
//	it will then set the window to the foreground and return.
bool WaitOnWindow(HWND &hWnd)
{
	while(!hWnd) //Wait on valid handle.
	{
		Sleep(1);
	}

    while(!IsWindowVisible(hWnd)) //Wait on window to be visible.
    {
        Sleep(1);
    }

    if(IsWindow(hWnd))
    {
        SetForegroundWindow(hWnd);
		return true;
    }

	return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//This function will wait for a specified amount of time for the
//	window specified by hWnd to become visible, it will then set
//	the window to the foreground and return.
bool WaitOnWindow(HWND &hWnd, int iMilliSeconds)
{
	while(!hWnd) //Wait on valid handle.
	{
		Sleep(1);
	}

	int iWaited = 0;
    while(!IsWindowVisible(hWnd))
    {
		if(iMilliSeconds > 0)
		{
			if(iWaited++ >= iMilliSeconds)
			{
				return false;
			}
		}
        Sleep(1);
    }

    if(IsWindow(hWnd))
    {
        SetForegroundWindow(hWnd);
		return true;
    }

	return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void ShowWindowInTaskbar(HWND hWnd, bool bShow)
{
	DWORD dwCurrentStyle = GetWindowLong(hWnd, GWL_EXSTYLE);
	if(bShow)
	{
		if(!(dwCurrentStyle &WS_EX_APPWINDOW))
		{
			SetWindowLong(hWnd, GWL_EXSTYLE, dwCurrentStyle | WS_EX_APPWINDOW);
		}
	}
	else{
		if((dwCurrentStyle &WS_EX_APPWINDOW))
		{
			SetWindowLong(hWnd, GWL_EXSTYLE, (GetWindowLong(hWnd, GWL_EXSTYLE) & ~WS_EX_APPWINDOW));
		}
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool SelectComboItem(HWND hCombo, const char *sText, int iMatchLength)
{
	LRESULT iIndex = -1;

	if(iMatchLength <= 0)
	{
		iIndex = SendMessage(hCombo, (UINT)CB_FINDSTRING, (WPARAM)-1, (LPARAM) sText);
	}
	else{
		char *sShortText = (char *) calloc(iMatchLength + 1, sizeof(char));
		strncpy_s(sShortText, iMatchLength + 1, sText, iMatchLength);
		iIndex = SendMessage(hCombo, (UINT)CB_FINDSTRING, (WPARAM)-1, (LPARAM) sShortText);
		free(sShortText);
	}

	if(iIndex >= 0)
	{
		return(SendMessage(hCombo, (UINT)CB_SETCURSEL, (WPARAM)iIndex, (LPARAM)NULL) == iIndex);
	}

	return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool SelectComboItem(HWND hCombo, const char *sText)
{
	return SelectComboItem(hCombo, sText, -1);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to show a validation message box on a Property Sheet control.

	If will move the focus to the complaining tab and stop the apply process.

	the Property sheet callback procedure should return TRUE if you call this function.

*/
int PropPageMsg(HWND hWnd, const char *sText, const char *sTitle, int iMsgBoxStyle)
{
	SendMessage(GetParent(hWnd), PSM_SETCURSEL, NULL, (LPARAM)hWnd);
	int iResult = MessageBox(hWnd, sText, sTitle, iMsgBoxStyle);
	SetWindowLongPtr(hWnd, DWLP_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
	return iResult;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	See [PropPageMsg] overload for notes.
*/
int PropPageMsg(HWND hWnd, const char *sText, const char *sTitle)
{
	return PropPageMsg(hWnd, sText, sTitle, MB_ICONINFORMATION);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#endif
